# frozen_string_literal: true

module Labkit
  ##
  # Fields is intended to be a SSOT for all of the common field names that
  # we emit via any observability we add to our systems.
  #
  # These fields should span multiple services.
  #
  # The goal of this package is to reduce the likelihood for typos or
  # subtly different naming conventions. This will help to ensure we
  # are able to marry up logs between different systems as a request
  # is being processed.
  #
  # Usage:
  #   require 'labkit/fields'
  #   ...
  #   data[Labkit::Fields::GL_USER_ID] = user.id
  #   ...
  #
  # Labkit (Go): https://gitlab.com/gitlab-org/labkit/-/tree/master/fields?ref_type=heads
  #
  # For Engineers Looking to add fields:
  #
  # These fields are derived from the Go Labkit variant. Please ensure that you've made the
  # respective changes in that repository prior to including the fields in this package.
  #
  # Please see the handbook page for more information
  # https://handbook.gitlab.com/handbook/engineering/architecture/design-documents/observability_field_standardisation/
  module Fields
    # correlation_id - string
    #
    # This field is used to correlate
    # the logs emitted by all of our systems.
    # This should be present in all log line
    # emissions.
    CORRELATION_ID = "correlation_id"

    # GitLabUserID - an integer field that
    # captures the user's numeric ID for logging purposes.
    GL_USER_ID = "gl_user_id"

    # GitLabUserName - a string field that
    # captures the user's username for logging purposes.
    GL_USER_NAME = "gl_user_name"

    # New fields being added to this section should have
    # the appropriate doc comments added above. These
    # should clearly indicate what the intended use of the
    # field is and should be replicated across the labkit
    # variations.

    # Get the constant name for a field value
    # @param field_value [String] The field value (e.g., "gl_user_id")
    # @return [String, nil] The constant name (e.g., "GL_USER_ID") or nil if not found
    def self.constant_name_for(field_value)
      constants(false).find do |const_name|
        next if const_name == :Deprecated

        const_get(const_name) == field_value
      end&.to_s
    end

    module Deprecated
      # This module tracks deprecated field names and maps them to their
      # standard replacements. These mappings are used by the field scanner
      # to identify and track usage of deprecated fields in the codebase.

      MAPPINGS = {
        Fields::GL_USER_ID => %w[user_id userid],
      }.freeze

      class << self
        # Get all deprecated fields as a lookup hash
        #
        # @return [Hash{String => String}] Hash mapping deprecated field names to standard field names
        def all
          @all ||= MAPPINGS.each_with_object({}) do |(key, values), result|
            values.each { |v| result[v] = key }
          end
        end

        # Check if a field is deprecated
        #
        # @param field_name [String, Symbol] The field name to check
        # @return [Boolean] true if the field is deprecated
        def deprecated?(field_name)
          all.key?(field_name.to_s)
        end

        # Get the standard field for a deprecated field
        #
        # @param deprecated_field [String, Symbol] The deprecated field name
        # @return [String, nil] The standard field name, or nil if not found
        def standard_field_for(deprecated_field)
          all[deprecated_field.to_s]
        end
      end
    end
  end
end
